// PixyTrade.h
// Header file for PixyTrade DLL - TradeStation EasyLanguage Integration
// Version 2.0

#pragma once

#ifdef PIXYTRADE_EXPORTS
#define PIXYTRADE_API __declspec(dllexport)
#else
#define PIXYTRADE_API __declspec(dllimport)
#endif

#include <string>
#include <map>
#include <mutex>
#include <thread>
#include <atomic>

// ============================================================
// SESSION DATA STRUCTURE
// ============================================================
struct SessionData {
    std::string symbol;
    std::string apiKey;
    std::atomic<bool> armed;
    std::atomic<bool> rthOnly;
    std::atomic<bool> valuesChanged;
    std::atomic<bool> isRunning;
    std::thread* pollThread;

    SessionData() : armed(false), rthOnly(true), valuesChanged(false),
                    isRunning(false), pollThread(nullptr) {}
};

// ============================================================
// EXPORTED FUNCTIONS (called by EasyLanguage)
// ============================================================
extern "C" {
    // Initialize session with symbol and API key
    // Returns session handle (>0) or 0 on error
    PIXYTRADE_API int __stdcall PT_Initialize(const char* symbol, const char* apiKey);

    // Get Armed status: returns 1=armed, 0=not armed, -1=error
    PIXYTRADE_API int __stdcall PT_GetArmed(int sessionHandle);

    // Get RTHOnly status: returns 1=RTH only, 0=extended hours, -1=error
    PIXYTRADE_API int __stdcall PT_GetRTHOnly(int sessionHandle);

    // Poll for updates: returns 1=values changed, 0=no change, -1=error
    PIXYTRADE_API int __stdcall PT_Poll(int sessionHandle);

    // Cleanup session
    PIXYTRADE_API void __stdcall PT_Cleanup(int sessionHandle);

    // Get last error message (for debugging)
    PIXYTRADE_API const char* __stdcall PT_GetLastError();
}

// ============================================================
// INTERNAL FUNCTIONS
// ============================================================
namespace PixyTrade {
    // Initialize the DLL (called once on load)
    void Initialize();

    // Cleanup all sessions (called on DLL unload)
    void Cleanup();

    // Call PixyTrade Scanner API
    bool CallScannerAPI(const std::string& symbol, const std::string& apiKey,
                        bool& armed, bool& rthOnly);

    // Background polling thread function
    void PollThreadFunc(int sessionHandle);

    // Set last error message
    void SetLastError(const std::string& error);
}
