// PixyTrade.cpp
// Implementation of PixyTrade DLL for TradeStation EasyLanguage Integration
// Version 2.0

#define PIXYTRADE_EXPORTS
#include "PixyTrade.h"

#include <windows.h>
#include <winhttp.h>
#include <string>
#include <map>
#include <mutex>
#include <thread>
#include <chrono>
#include <sstream>
#include <algorithm>

#pragma comment(lib, "winhttp.lib")

// ============================================================
// CONFIGURATION
// ============================================================
static const wchar_t* API_HOST = L"pixytrade.com";
static const wchar_t* API_PATH = L"/scanner/analyze";
static const int API_PORT = INTERNET_DEFAULT_HTTPS_PORT;
static const int POLL_INTERVAL_MS = 5000;  // Poll every 5 seconds

// Timeout configuration (in milliseconds)
static const int TIMEOUT_RESOLVE_MS = 5000;   // DNS resolution timeout
static const int TIMEOUT_CONNECT_MS = 5000;   // Connection timeout
static const int TIMEOUT_SEND_MS = 10000;     // Send timeout
static const int TIMEOUT_RECEIVE_MS = 10000;  // Receive timeout

// ============================================================
// GLOBAL STATE
// ============================================================
static std::map<int, SessionData*> g_sessions;
static std::mutex g_sessionMutex;
static int g_nextHandle = 1;
static std::string g_lastError;
static std::mutex g_errorMutex;
static bool g_initialized = false;

// ============================================================
// HELPER FUNCTIONS
// ============================================================

namespace PixyTrade {

void SetLastError(const std::string& error) {
    std::lock_guard<std::mutex> lock(g_errorMutex);
    g_lastError = error;
}

std::string GetLastErrorInternal() {
    std::lock_guard<std::mutex> lock(g_errorMutex);
    return g_lastError;
}

// Simple JSON value extraction (no external dependencies)
std::string ExtractJsonBool(const std::string& json, const std::string& key) {
    std::string searchKey = "\"" + key + "\"";
    size_t pos = json.find(searchKey);
    if (pos == std::string::npos) return "";

    // Find the colon after the key
    pos = json.find(':', pos);
    if (pos == std::string::npos) return "";
    pos++;

    // Skip whitespace
    while (pos < json.length() && (json[pos] == ' ' || json[pos] == '\t')) {
        pos++;
    }

    // Extract the value (true, false, or quoted string)
    if (json.substr(pos, 4) == "true") return "true";
    if (json.substr(pos, 5) == "false") return "false";

    return "";
}

std::wstring StringToWString(const std::string& str) {
    if (str.empty()) return std::wstring();
    int size = MultiByteToWideChar(CP_UTF8, 0, str.c_str(), -1, nullptr, 0);
    std::wstring wstr(size - 1, 0);
    MultiByteToWideChar(CP_UTF8, 0, str.c_str(), -1, &wstr[0], size);
    return wstr;
}

std::string WStringToString(const std::wstring& wstr) {
    if (wstr.empty()) return std::string();
    int size = WideCharToMultiByte(CP_UTF8, 0, wstr.c_str(), -1, nullptr, 0, nullptr, nullptr);
    std::string str(size - 1, 0);
    WideCharToMultiByte(CP_UTF8, 0, wstr.c_str(), -1, &str[0], size, nullptr, nullptr);
    return str;
}

bool CallScannerAPI(const std::string& symbol, const std::string& apiKey,
                    bool& armed, bool& rthOnly) {
    HINTERNET hSession = nullptr;
    HINTERNET hConnect = nullptr;
    HINTERNET hRequest = nullptr;
    bool success = false;

    // Default safe values
    armed = false;
    rthOnly = true;

    // Create session
    hSession = WinHttpOpen(L"PixyTrade/2.0",
                          WINHTTP_ACCESS_TYPE_DEFAULT_PROXY,
                          WINHTTP_NO_PROXY_NAME,
                          WINHTTP_NO_PROXY_BYPASS, 0);
    if (!hSession) {
        SetLastError("Failed to open WinHTTP session");
        return false;
    }

    // Set timeouts to prevent TradeStation lockup
    if (!WinHttpSetTimeouts(hSession,
                            TIMEOUT_RESOLVE_MS,   // DNS resolution
                            TIMEOUT_CONNECT_MS,   // Connection
                            TIMEOUT_SEND_MS,      // Send
                            TIMEOUT_RECEIVE_MS))  // Receive
    {
        SetLastError("Failed to set HTTP timeouts");
        WinHttpCloseHandle(hSession);
        return false;
    }

    // Connect to server
    hConnect = WinHttpConnect(hSession, API_HOST, API_PORT, 0);
    if (!hConnect) {
        SetLastError("Failed to connect to server");
        WinHttpCloseHandle(hSession);
        return false;
    }

    // Build the path with symbol parameter
    std::wstring path = API_PATH;
    path += L"?symbol=";
    path += StringToWString(symbol);

    // Create request
    hRequest = WinHttpOpenRequest(hConnect, L"GET", path.c_str(),
                                  nullptr, WINHTTP_NO_REFERER,
                                  WINHTTP_DEFAULT_ACCEPT_TYPES,
                                  WINHTTP_FLAG_SECURE);
    if (!hRequest) {
        SetLastError("Failed to create request");
        WinHttpCloseHandle(hConnect);
        WinHttpCloseHandle(hSession);
        return false;
    }

    // Add Authorization header
    std::wstring authHeader = L"Authorization: Bearer ";
    authHeader += StringToWString(apiKey);

    if (!WinHttpAddRequestHeaders(hRequest, authHeader.c_str(), (ULONG)-1,
                                   WINHTTP_ADDREQ_FLAG_ADD)) {
        SetLastError("Failed to add authorization header");
        WinHttpCloseHandle(hRequest);
        WinHttpCloseHandle(hConnect);
        WinHttpCloseHandle(hSession);
        return false;
    }

    // Send request
    if (!WinHttpSendRequest(hRequest, WINHTTP_NO_ADDITIONAL_HEADERS, 0,
                            WINHTTP_NO_REQUEST_DATA, 0, 0, 0)) {
        DWORD error = GetLastError();
        std::stringstream ss;
        ss << "Failed to send request (error " << error << ")";
        SetLastError(ss.str());
        WinHttpCloseHandle(hRequest);
        WinHttpCloseHandle(hConnect);
        WinHttpCloseHandle(hSession);
        return false;
    }

    // Receive response
    if (!WinHttpReceiveResponse(hRequest, nullptr)) {
        DWORD error = GetLastError();
        std::stringstream ss;
        ss << "Failed to receive response (error " << error << ")";
        SetLastError(ss.str());
        WinHttpCloseHandle(hRequest);
        WinHttpCloseHandle(hConnect);
        WinHttpCloseHandle(hSession);
        return false;
    }

    // Check status code
    DWORD statusCode = 0;
    DWORD statusCodeSize = sizeof(statusCode);
    WinHttpQueryHeaders(hRequest, WINHTTP_QUERY_STATUS_CODE | WINHTTP_QUERY_FLAG_NUMBER,
                        WINHTTP_HEADER_NAME_BY_INDEX, &statusCode, &statusCodeSize,
                        WINHTTP_NO_HEADER_INDEX);

    if (statusCode != 200) {
        std::stringstream ss;
        ss << "API returned status " << statusCode;
        SetLastError(ss.str());
        WinHttpCloseHandle(hRequest);
        WinHttpCloseHandle(hConnect);
        WinHttpCloseHandle(hSession);
        return false;
    }

    // Read response body
    std::string responseBody;
    DWORD bytesAvailable = 0;
    DWORD bytesRead = 0;
    char buffer[4096];

    while (WinHttpQueryDataAvailable(hRequest, &bytesAvailable) && bytesAvailable > 0) {
        DWORD toRead = (std::min)(bytesAvailable, (DWORD)(sizeof(buffer) - 1));
        if (WinHttpReadData(hRequest, buffer, toRead, &bytesRead)) {
            buffer[bytesRead] = '\0';
            responseBody += buffer;
        }
    }

    // Parse JSON response
    // Expected format: {"armed": true/false, "rthOnly": true/false}
    std::string armedStr = ExtractJsonBool(responseBody, "armed");
    std::string rthOnlyStr = ExtractJsonBool(responseBody, "rthOnly");

    if (armedStr.empty() && rthOnlyStr.empty()) {
        SetLastError("Invalid JSON response: " + responseBody);
    } else {
        armed = (armedStr == "true");
        rthOnly = (rthOnlyStr == "true");
        success = true;
    }

    // Cleanup
    WinHttpCloseHandle(hRequest);
    WinHttpCloseHandle(hConnect);
    WinHttpCloseHandle(hSession);

    return success;
}

void PollThreadFunc(int sessionHandle) {
    SessionData* session = nullptr;

    {
        std::lock_guard<std::mutex> lock(g_sessionMutex);
        auto it = g_sessions.find(sessionHandle);
        if (it != g_sessions.end()) {
            session = it->second;
        }
    }

    if (!session) return;

    while (session->isRunning.load()) {
        bool newArmed = false;
        bool newRthOnly = true;

        if (CallScannerAPI(session->symbol, session->apiKey, newArmed, newRthOnly)) {
            // Check if values changed
            bool oldArmed = session->armed.load();
            bool oldRthOnly = session->rthOnly.load();

            if (newArmed != oldArmed || newRthOnly != oldRthOnly) {
                session->armed.store(newArmed);
                session->rthOnly.store(newRthOnly);
                session->valuesChanged.store(true);
            }
        }

        // Sleep in small intervals to allow quick shutdown
        for (int i = 0; i < POLL_INTERVAL_MS / 100 && session->isRunning.load(); i++) {
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
        }
    }
}

void Initialize() {
    if (g_initialized) return;
    g_initialized = true;
}

void Cleanup() {
    std::lock_guard<std::mutex> lock(g_sessionMutex);
    for (auto& pair : g_sessions) {
        SessionData* session = pair.second;
        if (session) {
            session->isRunning.store(false);
            if (session->pollThread && session->pollThread->joinable()) {
                session->pollThread->join();
            }
            delete session->pollThread;
            delete session;
        }
    }
    g_sessions.clear();
    g_initialized = false;
}

} // namespace PixyTrade

// ============================================================
// DLL ENTRY POINT
// ============================================================
BOOL APIENTRY DllMain(HMODULE hModule, DWORD ul_reason_for_call, LPVOID lpReserved) {
    switch (ul_reason_for_call) {
    case DLL_PROCESS_ATTACH:
        DisableThreadLibraryCalls(hModule);
        PixyTrade::Initialize();
        break;
    case DLL_PROCESS_DETACH:
        PixyTrade::Cleanup();
        break;
    }
    return TRUE;
}

// ============================================================
// EXPORTED FUNCTIONS
// ============================================================

extern "C" {

PIXYTRADE_API int __stdcall PT_Initialize(const char* symbol, const char* apiKey) {
    if (!symbol || !apiKey || strlen(symbol) == 0 || strlen(apiKey) == 0) {
        PixyTrade::SetLastError("Symbol and API key are required");
        return 0;
    }

    SessionData* session = new SessionData();
    session->symbol = symbol;
    session->apiKey = apiKey;

    // Uppercase the symbol
    for (auto& c : session->symbol) {
        c = toupper(c);
    }

    // Do initial API call
    bool armed = false;
    bool rthOnly = true;
    if (!PixyTrade::CallScannerAPI(session->symbol, session->apiKey, armed, rthOnly)) {
        // Continue anyway, the background thread will retry
    }
    session->armed.store(armed);
    session->rthOnly.store(rthOnly);

    int handle;
    {
        std::lock_guard<std::mutex> lock(g_sessionMutex);
        handle = g_nextHandle++;
        g_sessions[handle] = session;
    }

    // Start background polling thread
    session->isRunning.store(true);
    session->pollThread = new std::thread(PixyTrade::PollThreadFunc, handle);

    return handle;
}

PIXYTRADE_API int __stdcall PT_GetArmed(int sessionHandle) {
    std::lock_guard<std::mutex> lock(g_sessionMutex);
    auto it = g_sessions.find(sessionHandle);
    if (it == g_sessions.end()) {
        return -1;
    }
    return it->second->armed.load() ? 1 : 0;
}

PIXYTRADE_API int __stdcall PT_GetRTHOnly(int sessionHandle) {
    std::lock_guard<std::mutex> lock(g_sessionMutex);
    auto it = g_sessions.find(sessionHandle);
    if (it == g_sessions.end()) {
        return -1;
    }
    return it->second->rthOnly.load() ? 1 : 0;
}

PIXYTRADE_API int __stdcall PT_Poll(int sessionHandle) {
    std::lock_guard<std::mutex> lock(g_sessionMutex);
    auto it = g_sessions.find(sessionHandle);
    if (it == g_sessions.end()) {
        return -1;
    }

    bool changed = it->second->valuesChanged.exchange(false);
    return changed ? 1 : 0;
}

PIXYTRADE_API void __stdcall PT_Cleanup(int sessionHandle) {
    SessionData* session = nullptr;

    {
        std::lock_guard<std::mutex> lock(g_sessionMutex);
        auto it = g_sessions.find(sessionHandle);
        if (it == g_sessions.end()) {
            return;
        }
        session = it->second;
        g_sessions.erase(it);
    }

    if (session) {
        session->isRunning.store(false);
        if (session->pollThread && session->pollThread->joinable()) {
            session->pollThread->join();
        }
        delete session->pollThread;
        delete session;
    }
}

PIXYTRADE_API const char* __stdcall PT_GetLastError() {
    static std::string errorCopy;
    errorCopy = PixyTrade::GetLastErrorInternal();
    return errorCopy.c_str();
}

} // extern "C"
